<?php

namespace App\Http\Controllers;

use App\Models\Checklist;
use App\Models\Equipment;
use App\Models\Inspection;
use App\Models\InspectionItem;
use App\Models\InspectionPhoto;
use App\Models\Setting;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class InspectionController extends Controller
{
    public function adminIndex(Request $request)
    {
        $query = Inspection::with(['user', 'equipment', 'checklist']);

        if ($request->filled('date_from')) {
            $query->whereDate('inspection_date', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('inspection_date', '<=', $request->date_to);
        }

        if ($request->filled('user_id')) {
            $query->where('user_id', $request->user_id);
        }

        if ($request->filled('equipment_id')) {
            $query->where('equipment_id', $request->equipment_id);
        }

        if ($request->filled('status')) {
            $query->where('overall_status', $request->status);
        }

        $inspections = $query->orderBy('created_at', 'desc')->paginate(20);

        $users = \App\Models\User::where('status', 'approved')->get();
        $equipment = Equipment::where('is_active', true)->get();

        return view('admin.inspections.index', compact('inspections', 'users', 'equipment'));
    }

    public function adminShow(Inspection $inspection)
    {
        $inspection->load(['user', 'equipment', 'checklist', 'items.checklistItem', 'photos']);
        return view('admin.inspections.show', compact('inspection'));
    }

    public function workerIndex()
    {
        $inspections = Inspection::with(['equipment', 'checklist'])
            ->where('user_id', auth()->id())
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return view('worker.inspections.index', compact('inspections'));
    }

    public function workerShow(Inspection $inspection)
    {
        if ($inspection->user_id !== auth()->id()) {
            abort(403, 'You can only view your own inspections.');
        }

        $inspection->load(['user', 'equipment', 'checklist', 'items.checklistItem', 'photos']);
        return view('worker.inspections.show', compact('inspection'));
    }

    public function selectChecklist()
    {
        $checklists = auth()->user()->checklists()
            ->where('is_active', true)
            ->withCount('items')
            ->get();

        return view('worker.inspections.select-checklist', compact('checklists'));
    }

    public function selectEquipment(Checklist $checklist)
    {
        if (!auth()->user()->checklists->contains($checklist)) {
            abort(403, 'You do not have access to this checklist.');
        }

        $equipment = $checklist->equipment()
            ->where('is_active', true)
            ->get();

        $inspectedRecently = [];
        foreach ($equipment as $equip) {
            $lastInspection = Inspection::where('equipment_id', $equip->id)
                ->where('checklist_id', $checklist->id)
                ->orderBy('inspection_date', 'desc')
                ->first();

            if ($lastInspection) {
                $daysSince = Carbon::parse($lastInspection->inspection_date)->diffInDays(now());
                if ($daysSince < $checklist->frequency_days) {
                    $inspectedRecently[$equip->id] = [
                        'last_inspection' => $lastInspection,
                        'days_remaining' => $checklist->frequency_days - $daysSince,
                    ];
                }
            }
        }

        return view('worker.inspections.select-equipment', compact('checklist', 'equipment', 'inspectedRecently'));
    }

    public function create(Checklist $checklist, Equipment $equipment)
    {
        if (!auth()->user()->checklists->contains($checklist)) {
            abort(403, 'You do not have access to this checklist.');
        }

        if (!$checklist->equipment->contains($equipment)) {
            abort(403, 'This equipment is not linked to the selected checklist.');
        }

        $lastInspection = Inspection::where('equipment_id', $equipment->id)
            ->where('checklist_id', $checklist->id)
            ->orderBy('inspection_date', 'desc')
            ->first();

        if ($lastInspection) {
            $daysSince = Carbon::parse($lastInspection->inspection_date)->diffInDays(now());
            if ($daysSince < $checklist->frequency_days) {
                return redirect()->route('worker.inspections.select-equipment', $checklist)
                    ->with('error', 'This equipment was inspected recently. Please wait ' . ($checklist->frequency_days - $daysSince) . ' more day(s).');
            }
        }

        $checklist->load('items');

        return view('worker.inspections.create', compact('checklist', 'equipment'));
    }

    public function store(Request $request, Checklist $checklist, Equipment $equipment)
    {
        if (!auth()->user()->checklists->contains($checklist)) {
            abort(403, 'You do not have access to this checklist.');
        }

        $lastInspection = Inspection::where('equipment_id', $equipment->id)
            ->where('checklist_id', $checklist->id)
            ->orderBy('inspection_date', 'desc')
            ->first();

        if ($lastInspection) {
            $daysSince = Carbon::parse($lastInspection->inspection_date)->diffInDays(now());
            if ($daysSince < $checklist->frequency_days) {
                return redirect()->route('worker.inspections.select-equipment', $checklist)
                    ->with('error', 'This equipment was inspected recently.');
            }
        }

        $checklist->load('items');

        $validated = $request->validate([
            'items' => 'required|array',
            'items.*.result' => 'required|in:ok,fail,na',
            'items.*.fail_description' => 'nullable|string',
            'notes' => 'nullable|string',
            'photos' => 'nullable|array',
            'photos.*' => 'image|max:5120',
        ]);

        foreach ($checklist->items as $item) {
            $itemData = $validated['items'][$item->id] ?? null;
            if (!$itemData) {
                return redirect()->back()
                    ->with('error', 'All checklist items must be completed.')
                    ->withInput();
            }

            if ($itemData['result'] === 'fail' && $item->require_description_on_fail && empty($itemData['fail_description'])) {
                return redirect()->back()
                    ->with('error', 'Failed items require a description: ' . $item->item_name)
                    ->withInput();
            }
        }

        $hasFailed = false;
        foreach ($validated['items'] as $itemData) {
            if ($itemData['result'] === 'fail') {
                $hasFailed = true;
                break;
            }
        }

        $inspection = Inspection::create([
            'user_id' => auth()->id(),
            'checklist_id' => $checklist->id,
            'equipment_id' => $equipment->id,
            'overall_status' => $hasFailed ? 'failed' : 'passed',
            'notes' => $validated['notes'] ?? null,
            'inspection_date' => now(),
        ]);

        foreach ($checklist->items as $item) {
            $itemData = $validated['items'][$item->id];
            InspectionItem::create([
                'inspection_id' => $inspection->id,
                'checklist_item_id' => $item->id,
                'result' => $itemData['result'],
                'fail_description' => $itemData['fail_description'] ?? null,
            ]);
        }

        if ($request->hasFile('photos')) {
            foreach ($request->file('photos') as $photo) {
                $filename = Str::uuid() . '.' . $photo->getClientOriginalExtension();
                $path = $photo->storeAs('inspection-photos', $filename, 'public');

                InspectionPhoto::create([
                    'inspection_id' => $inspection->id,
                    'filename' => $filename,
                    'original_name' => $photo->getClientOriginalName(),
                    'path' => $path,
                    'size' => $photo->getSize(),
                    'mime_type' => $photo->getMimeType(),
                ]);
            }
        }

        $this->sendNotificationEmail($inspection);

        return redirect()->route('worker.inspections.show', $inspection)
            ->with('success', 'Inspection submitted successfully.');
    }

    private function sendNotificationEmail(Inspection $inspection)
    {
        $safetyManagerEmail = Setting::get('safety_manager_email');
        
        if (!$safetyManagerEmail) {
            return;
        }

        try {
            Mail::send('emails.inspection-notification', [
                'inspection' => $inspection,
                'url' => route('admin.inspections.show', $inspection),
            ], function ($message) use ($safetyManagerEmail, $inspection) {
                $message->to($safetyManagerEmail)
                    ->subject('New Inspection Report - ' . $inspection->equipment->name);
            });
        } catch (\Exception $e) {
            \Log::error('Failed to send inspection notification email: ' . $e->getMessage());
        }
    }

    public function printReport(Inspection $inspection)
    {
        if (auth()->user()->isWorker() && $inspection->user_id !== auth()->id()) {
            abort(403, 'You can only view your own inspection reports.');
        }

        $inspection->load(['user.department', 'equipment.department', 'checklist', 'items.checklistItem', 'photos']);
        
        return view('inspections.print', compact('inspection'));
    }

    public function export(Request $request)
    {
        $query = Inspection::with(['user', 'equipment', 'checklist']);

        if ($request->filled('date_from')) {
            $query->whereDate('inspection_date', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('inspection_date', '<=', $request->date_to);
        }

        if ($request->filled('status')) {
            $query->where('overall_status', $request->status);
        }

        $inspections = $query->orderBy('inspection_date', 'desc')->get();

        $format = $request->get('format', 'csv');

        if ($format === 'csv') {
            return $this->exportCsv($inspections);
        }

        return $this->exportHtml($inspections);
    }

    private function exportCsv($inspections)
    {
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="inspections-' . date('Y-m-d') . '.csv"',
        ];

        $callback = function () use ($inspections) {
            $file = fopen('php://output', 'w');
            
            fputcsv($file, [
                'ID', 'Date', 'Worker', 'Employee ID', 'Equipment', 'Tag ID',
                'Checklist', 'Status', 'Notes'
            ]);

            foreach ($inspections as $inspection) {
                fputcsv($file, [
                    $inspection->id,
                    $inspection->inspection_date->format('Y-m-d H:i:s'),
                    $inspection->user->full_name,
                    $inspection->user->employee_id,
                    $inspection->equipment->name,
                    $inspection->equipment->tag_id,
                    $inspection->checklist->title,
                    $inspection->overall_status,
                    $inspection->notes,
                ]);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }

    private function exportHtml($inspections)
    {
        return view('admin.inspections.export', compact('inspections'));
    }
}
