<?php

namespace App\Http\Controllers;

use App\Models\Checklist;
use App\Models\ChecklistItem;
use App\Models\Equipment;
use Illuminate\Http\Request;

class ChecklistController extends Controller
{
    public function index()
    {
        $checklists = Checklist::withCount(['items', 'equipment', 'users'])
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return view('admin.checklists.index', compact('checklists'));
    }

    public function create()
    {
        $equipment = Equipment::where('is_active', true)->get();
        return view('admin.checklists.create', compact('equipment'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'frequency_days' => 'required|integer|min:1|max:365',
            'is_active' => 'boolean',
            'equipment' => 'array',
            'equipment.*' => 'exists:equipment,id',
            'items' => 'array',
            'items.*.item_name' => 'required|string|max:255',
            'items.*.description' => 'nullable|string',
            'items.*.require_description_on_fail' => 'boolean',
        ]);

        $checklist = Checklist::create([
            'title' => $validated['title'],
            'description' => $validated['description'] ?? null,
            'frequency_days' => $validated['frequency_days'],
            'is_active' => $request->boolean('is_active', true),
        ]);

        if (!empty($validated['equipment'])) {
            $checklist->equipment()->sync($validated['equipment']);
        }

        if (!empty($validated['items'])) {
            foreach ($validated['items'] as $index => $item) {
                ChecklistItem::create([
                    'checklist_id' => $checklist->id,
                    'item_name' => $item['item_name'],
                    'description' => $item['description'] ?? null,
                    'require_description_on_fail' => !empty($item['require_description_on_fail']),
                    'sort_order' => $index,
                ]);
            }
        }

        return redirect()->route('admin.checklists.index')
            ->with('success', 'Checklist created successfully.');
    }

    public function show(Checklist $checklist)
    {
        $checklist->load(['items', 'equipment', 'users']);
        return view('admin.checklists.show', compact('checklist'));
    }

    public function edit(Checklist $checklist)
    {
        $checklist->load('items', 'equipment');
        $equipment = Equipment::where('is_active', true)->get();
        $selectedEquipment = $checklist->equipment->pluck('id')->toArray();

        return view('admin.checklists.edit', compact('checklist', 'equipment', 'selectedEquipment'));
    }

    public function update(Request $request, Checklist $checklist)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'frequency_days' => 'required|integer|min:1|max:365',
            'is_active' => 'boolean',
            'equipment' => 'array',
            'equipment.*' => 'exists:equipment,id',
        ]);

        $checklist->update([
            'title' => $validated['title'],
            'description' => $validated['description'] ?? null,
            'frequency_days' => $validated['frequency_days'],
            'is_active' => $request->boolean('is_active', true),
        ]);

        $checklist->equipment()->sync($validated['equipment'] ?? []);

        return redirect()->route('admin.checklists.index')
            ->with('success', 'Checklist updated successfully.');
    }

    public function destroy(Checklist $checklist)
    {
        $checklist->delete();

        return redirect()->route('admin.checklists.index')
            ->with('success', 'Checklist deleted successfully.');
    }

    public function storeItem(Request $request, Checklist $checklist)
    {
        $validated = $request->validate([
            'item_name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'require_description_on_fail' => 'boolean',
        ]);

        $maxOrder = $checklist->items()->max('sort_order') ?? 0;

        ChecklistItem::create([
            'checklist_id' => $checklist->id,
            'item_name' => $validated['item_name'],
            'description' => $validated['description'] ?? null,
            'require_description_on_fail' => $request->boolean('require_description_on_fail'),
            'sort_order' => $maxOrder + 1,
        ]);

        return redirect()->back()->with('success', 'Item added successfully.');
    }

    public function updateItem(Request $request, ChecklistItem $item)
    {
        $validated = $request->validate([
            'item_name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'require_description_on_fail' => 'boolean',
        ]);

        $item->update([
            'item_name' => $validated['item_name'],
            'description' => $validated['description'] ?? null,
            'require_description_on_fail' => $request->boolean('require_description_on_fail'),
        ]);

        return redirect()->back()->with('success', 'Item updated successfully.');
    }

    public function destroyItem(ChecklistItem $item)
    {
        $item->delete();
        return redirect()->back()->with('success', 'Item deleted successfully.');
    }
}
