<?php

namespace App\Http\Controllers;

use App\Models\Checklist;
use App\Models\Department;
use App\Models\Equipment;
use App\Models\Inspection;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Http\Request;

class AdminController extends Controller
{
    public function dashboard()
    {
        $stats = [
            'total_users' => User::count(),
            'pending_users' => User::where('status', 'pending')->count(),
            'total_equipment' => Equipment::count(),
            'total_checklists' => Checklist::count(),
            'total_inspections' => Inspection::count(),
            'passed_inspections' => Inspection::where('overall_status', 'passed')->count(),
            'failed_inspections' => Inspection::where('overall_status', 'failed')->count(),
        ];

        $recentInspections = Inspection::with(['user', 'equipment', 'checklist'])
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get();

        $pendingUsers = User::with('department')
            ->where('status', 'pending')
            ->orderBy('created_at', 'desc')
            ->limit(5)
            ->get();

        $overdueEquipment = $this->getOverdueEquipment();

        return view('admin.dashboard', compact('stats', 'recentInspections', 'pendingUsers', 'overdueEquipment'));
    }

    private function getOverdueEquipment()
    {
        $equipment = Equipment::with(['checklists', 'inspections' => function($query) {
            $query->orderBy('inspection_date', 'desc');
        }])->where('is_active', true)->get();

        $overdue = [];
        foreach ($equipment as $equip) {
            foreach ($equip->checklists as $checklist) {
                $lastInspection = $equip->inspections
                    ->where('checklist_id', $checklist->id)
                    ->first();

                if (!$lastInspection) {
                    $overdue[] = [
                        'equipment' => $equip,
                        'checklist' => $checklist,
                        'days_overdue' => 'Never inspected',
                    ];
                } else {
                    $daysSince = Carbon::parse($lastInspection->inspection_date)->diffInDays(now());
                    if ($daysSince > $checklist->frequency_days) {
                        $overdue[] = [
                            'equipment' => $equip,
                            'checklist' => $checklist,
                            'days_overdue' => $daysSince - $checklist->frequency_days,
                        ];
                    }
                }
            }
        }

        return collect($overdue)->take(10);
    }

    public function users()
    {
        $users = User::with('department', 'checklists')
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return view('admin.users.index', compact('users'));
    }

    public function pendingUsers()
    {
        $users = User::with('department')
            ->where('status', 'pending')
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        $checklists = Checklist::where('is_active', true)->get();

        return view('admin.users.pending', compact('users', 'checklists'));
    }

    public function approveUser(Request $request, User $user)
    {
        $validated = $request->validate([
            'checklists' => 'array',
            'checklists.*' => 'exists:checklists,id',
        ]);

        $user->update(['status' => 'approved']);
        
        if (!empty($validated['checklists'])) {
            $user->checklists()->sync($validated['checklists']);
        }

        return redirect()->back()->with('success', 'User approved successfully.');
    }

    public function rejectUser(User $user)
    {
        $user->update(['status' => 'rejected']);
        return redirect()->back()->with('success', 'User rejected.');
    }

    public function editUser(User $user)
    {
        $departments = Department::where('is_active', true)->get();
        $checklists = Checklist::where('is_active', true)->get();
        $userChecklists = $user->checklists->pluck('id')->toArray();

        return view('admin.users.edit', compact('user', 'departments', 'checklists', 'userChecklists'));
    }

    public function updateUser(Request $request, User $user)
    {
        $validated = $request->validate([
            'full_name' => 'required|string|max:255',
            'username' => 'required|string|max:255|unique:users,username,' . $user->id,
            'employee_id' => 'required|string|max:255|unique:users,employee_id,' . $user->id,
            'department_id' => 'required|exists:departments,id',
            'email' => 'required|email|max:255|unique:users,email,' . $user->id,
            'role' => 'required|in:admin,worker',
            'status' => 'required|in:pending,approved,rejected',
            'checklists' => 'array',
            'checklists.*' => 'exists:checklists,id',
        ]);

        $user->update([
            'full_name' => $validated['full_name'],
            'username' => $validated['username'],
            'employee_id' => $validated['employee_id'],
            'department_id' => $validated['department_id'],
            'email' => $validated['email'],
            'role' => $validated['role'],
            'status' => $validated['status'],
        ]);

        $user->checklists()->sync($validated['checklists'] ?? []);

        return redirect()->route('admin.users.index')->with('success', 'User updated successfully.');
    }

    public function deleteUser(User $user)
    {
        if ($user->id === auth()->id()) {
            return redirect()->back()->with('error', 'You cannot delete your own account.');
        }

        $user->delete();
        return redirect()->back()->with('success', 'User deleted successfully.');
    }
}
